<?php
/** NOTICE OF LICENSE
 *
 * This module was created by veebipoed.ee and is protected by the laws of Copyright.
 * This use license is granted for only one website.
 * To use this module on several websites, you must purchase as many licenses as websites on which you want to use it.
 * Use, copy, modification or distribution of this module without
 * written license agreement from veebipoed.ee is strictly forbidden.
 * In order to obtain a license, please contact us: info@veebipoed.ee
 * Any infringement of these provisions as well as general copyrights will be prosecuted.
 * ...........................................................................
 *
 *
 * @author     VEEBIPOED.EE
 * @copyright  Copyright (c) 2012-2021 veebipoed.ee (http://www.veebipoed.ee)
 * @license    Commercial license
 * Support by mail: info@veebipoed.ee
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

require_once(dirname(__FILE__).'/classes/PlacetApplication.php');

class PlacetGroup extends PaymentModule
{
    protected $_html = '';
    protected $_postErrors = array();

    public $details;
    public $owner;
    public $address;
    public $extra_mail_vars;
    
    /** @var int EU compliance flag */
    public $is_eu_compatible;
    
    /** @var array Module controllers */
    public $controllers;
    
    /** @var bool Currency support */
    public $currencies;
    
    /** @var string Currency mode */
    public $currencies_mode;
    
    /** @var bool Bootstrap mode */
    public $bootstrap;

    /** @var int Module is configurable */
    public $is_configurable;

    /** @var int Module needs instance */
    public $need_instance;

    public $payment_method_label_template = "Placet Jarelmaks (Id Cart: %d)";

    public function __construct()
    {
        $this->name = 'placetgroup';
        $this->tab = 'payments_gateways';
        $this->version = '2.0.0';
        $this->ps_versions_compliancy = array('min' => '1.7.0', 'max' => '9.99.99');
        $this->author = 'Truetech';
        $this->controllers = array('validation');
        $this->is_eu_compatible = 1;

        $this->currencies = true;
        $this->currencies_mode = 'checkbox';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = $this->l('Placet Group');
        $this->description = $this->l('Placet Group Description');
    }

    public function install()
    {
        if (is_null($this->warning) && !function_exists('curl_init')) {
            if ($this->l('ERROR_MESSAGE_CURL_REQUIRED') == "ERROR_MESSAGE_CURL_REQUIRED") {
                $this->warning = "cURL is required to use this module. Please install the php extention cURL.";
            } else {
                $this->warning = $this->l('ERROR_MESSAGE_CURL_REQUIRED');
            }
        }

        $sqlResult = include(dirname(__FILE__).'/sql/install.php');

        if (!parent::install() ||
            !$sqlResult ||
            !$this->registerHook('header') ||
            !$this->registerHook('paymentOptions') ||
            !$this->registerHook('paymentReturn') ||
            !$this->registerHook('displayPaymentReturn') ||
            !$this->createOrderStates()
        ) {
            return false;
        }
        return true;
    }

    public function uninstall(){
        if ( !$this->unregisterHook('header') ||
            !$this->unregisterHook('paymentReturn')
            || !$this->unregisterHook('displayPaymentReturn')
            || !$this->unregisterHook('paymentOptions')
            || !parent::uninstall()) {
            return false;
        }
        return true;
    }
    
    public function hookHeader(){
        // Load CSS only on checkout page
        if ($this->context->controller instanceof OrderController) {
            $this->context->controller->addCSS($this->_path.'views/css/front.css', 'all');
        }
    }

    public function hookPaymentOptions($params)
    {
        if (!$this->active) {
            return;
        }

        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        $payment_options = [
            //$this->getOfflinePaymentOption(),
            $this->getExternalPaymentOption(),
            //$this->getEmbeddedPaymentOption(),
            //$this->getIframePaymentOption(),
        ];

        return $payment_options;
    }

    public function hookPaymentReturn($params)
    {
        if (!$this->active) {
            return;
        }

        // PrestaShop 8.x uses 'order' instead of 'objOrder'
        $order = isset($params['order']) ? $params['order'] : (isset($params['objOrder']) ? $params['objOrder'] : null);
        
        if (!$order || !Validate::isLoadedObject($order)) {
            return;
        }

        $state = $order->getCurrentState();
        if (in_array($state, array(Configuration::get('PS_OS_PAYMENT'))))
        {
            // PrestaShop 8.x uses 'currency' instead of 'currencyObj'
            $currency = isset($params['currency']) ? $params['currency'] : (isset($params['currencyObj']) ? $params['currencyObj'] : null);
            $totalToPay = isset($params['total_to_pay']) ? $params['total_to_pay'] : $order->total_paid;
            
            $this->smarty->assign(array(
                'shop_name' => $this->context->shop->name,
                'total_to_pay' => Tools::displayPrice($totalToPay, $currency, false),
                'status' => 'ok',
                'id_order' => $order->id
            ));
            if (isset($order->reference) && !empty($order->reference))
                $this->smarty->assign('reference', $order->reference);
        }
        else {
            $this->smarty->assign('status', 'failed');
        }
        return $this->display(__FILE__, 'payment_return.tpl');
    }

    public function hookDisplayPaymentReturn($params)
    {
        return $this->hookPaymentReturn($params);
    }

    public function checkCurrency($cart)
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);

        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }
        return false;
    }

    public function getOfflinePaymentOption()
    {
        $offlineOption = new PaymentOption();
        $offlineOption->setCallToActionText($this->l('Pay offline'))
                      ->setAction($this->context->link->getModuleLink($this->name, 'validation', array(), true))
                      ->setAdditionalInformation($this->context->smarty->fetch('module:placetgroup/views/templates/front/payment_infos.tpl'))
                      ->setLogo(Media::getMediaPath(_PS_MODULE_DIR_.$this->name.'/payment.jpg'));

        return $offlineOption;
    }

    public function getExternalPaymentOption()
    {
        $externalOption = new PaymentOption();
        $externalOption/*->setCallToActionText($this->l('Placet Group'))*/
                       ->setAction($this->context->link->getModuleLink($this->name, 'validation', array(), true))
                       ->setInputs([
                            'token' => [
                                'name' =>'token',
                                'type' =>'hidden',
                                'value' =>'12345689',
                            ],
                        ])
                       ->setAdditionalInformation($this->context->smarty->fetch('module:placetgroup/views/templates/front/payment_infos.tpl'))
                       ->setLogo(Media::getMediaPath(_PS_MODULE_DIR_.$this->name.'/payment.jpg'));

        return $externalOption;
    }

    public function getEmbeddedPaymentOption()
    {
        $embeddedOption = new PaymentOption();
        $embeddedOption->setCallToActionText($this->l('Pay embedded'))
                       ->setForm($this->generateForm())
                       ->setAdditionalInformation($this->context->smarty->fetch('module:placetgroup/views/templates/front/payment_infos.tpl'))
                       ->setLogo(Media::getMediaPath(_PS_MODULE_DIR_.$this->name.'/payment.jpg'));

        return $embeddedOption;
    }

    public function getIframePaymentOption()
    {
        $iframeOption = new PaymentOption();
        $iframeOption->setCallToActionText($this->l('Pay iframe'))
                     ->setAction($this->context->link->getModuleLink($this->name, 'iframe', array(), true))
                     ->setAdditionalInformation($this->context->smarty->fetch('module:placetgroup/views/templates/front/payment_infos.tpl'))
                     ->setLogo(Media::getMediaPath(_PS_MODULE_DIR_.$this->name.'/payment.jpg'));

        return $iframeOption;
    }

    /**
     * hookPayment - REMOVED for PrestaShop 8.x compatibility
     * This hook is deprecated since PS 1.7 and removed in PS 8.x
     * Use hookPaymentOptions instead
     */

    protected function generateForm()
    {
        $months = [];
        for ($i = 1; $i <= 12; $i++) {
            $months[] = sprintf("%02d", $i);
        }

        $years = [];
        for ($i = 0; $i <= 10; $i++) {
            $years[] = date('Y', strtotime('+'.$i.' years'));
        }

        $this->context->smarty->assign([
            'action' => $this->context->link->getModuleLink($this->name, 'validation', array(), true),
            'months' => $months,
            'years' => $years,
        ]);

        return $this->context->smarty->fetch('module:placetgroup/views/templates/front/payment_form.tpl');
    }

    public function getContent()
    {
        $output = null;

        if (Tools::isSubmit('submit'.$this->name)) {
            $myModuleName = strval(Tools::getValue('placetgroup'));
            $partnerAlias = strval(Tools::getValue('placetgroup_partneralias'));

            if (
                !$myModuleName || empty($myModuleName) || !Validate::isGenericName($myModuleName)
                || empty($partnerAlias) || !Validate::isGenericName($partnerAlias)
            ) {
                $output .= $this->displayError($this->l('Invalid Configuration value'));
            } else {
                Configuration::updateValue('placetgroup', $myModuleName);
                Configuration::updateValue('placetgroup_partneralias', $partnerAlias);
                $output .= $this->displayConfirmation($this->l('Settings updated'));
            }
        }

        return $output.$this->displayForm();
    }

    public function displayForm()
    {
        // Get default language
        $defaultLang = (int)Configuration::get('PS_LANG_DEFAULT');

        // Init Fields form array
        $fieldsForm[0]['form'] = [
            'legend' => [
                'title' => $this->l('Settings'),
            ],
            'input' => [
                [
                    'type' => 'text',
                    'label' => $this->l('Placet URL'),
                    'name' => 'placetgroup',
                    'size' => 20,
                    'required' => true,
                    'desc' => $this->l('Placet service URL withot trailing slashes, e.g: https://web-partner.placet.ee'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->l('Partner Alias'),
                    'name' => 'placetgroup_partneralias',
                    'size' => 20,
                    'required' => true
                ]
            ],
            'submit' => [
                'title' => $this->l('Save'),
                'class' => 'btn btn-default pull-right'
            ]
        ];

        $helper = new HelperForm();

        // Module, token and currentIndex
        $helper->module = $this;
        $helper->name_controller = $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

        // Language
        $helper->default_form_language = $defaultLang;
        $helper->allow_employee_form_lang = $defaultLang;

        // Title and toolbar
        $helper->title = $this->displayName;
        $helper->show_toolbar = true;        // false -> remove toolbar
        $helper->toolbar_scroll = true;      // yes - > Toolbar is always visible on the top of the screen.
        $helper->submit_action = 'submit'.$this->name;
        $helper->toolbar_btn = [
            'save' => [
                'desc' => $this->l('Save'),
                'href' => AdminController::$currentIndex.'&configure='.$this->name.'&save'.$this->name.
                    '&token='.Tools::getAdminTokenLite('AdminModules'),
            ],
            'back' => [
                'href' => AdminController::$currentIndex.'&token='.Tools::getAdminTokenLite('AdminModules'),
                'desc' => $this->l('Back to list')
            ]
        ];

        // Load current value
        $helper->fields_value['placetgroup'] = Tools::getValue('placetgroup', Configuration::get('placetgroup'));
        $helper->fields_value['placetgroup_partneralias'] = Configuration::get('placetgroup_partneralias');

        return $helper->generateForm($fieldsForm);
    }

    public function createApplicationRecord($token, $idCart, $cartTotal, $idCurrency, $idOrder = false)
    {
        if (is_object($token) && isset($token->token)) {
            $token = (string)$token->token;
        } else {
            $token = (string)$token;
        }
        $token = pSQL($token);
        $idCart =  (int)$idCart;
        $cartTotal =  (float)$cartTotal;
        $idCurrency =  (int)$idCurrency;
        $idOrder =  (int)$idOrder;
        $result = false;
        $errorMessage = 'Error creating PlacetGroup application record';
        do {
            if (empty($token)){
                $errorMessage .= sprintf(
                    ' - Missing token data: %s | %s',
                    $this->getVarFormat($idCart, 'id_cart'),
                    $this->getVarFormat($cartTotal, 'cart_total')
                );
                break;
            }

            if (empty($idCart) || empty($cartTotal) || empty($idCurrency)){
                $errorMessage .= sprintf(
                    ' - Missing cart data: %s | %s | %s',
                    $this->getVarFormat($idCart, 'id_cart'),
                    $this->getVarFormat($cartTotal, 'cart_total'),
                    $this->getVarFormat($idCurrency, 'id_currency')
                );
                break;
            }

            if (!empty($idApplication = PlacetApplication::getIdByToken($token))){
                $errorMessage .= sprintf(
                    ' - Application record with this token already exists: %s | %s',
                    $this->getVarFormat($idCart, 'id_cart'),
                    $this->getVarFormat($cartTotal, 'cart_total')
                );
                break;
            }

            $placetApplication = new PlacetApplication();

            $placetApplication->token = $token;
            $placetApplication->id_cart = $idCart;
            $placetApplication->cart_total = $cartTotal;
            $placetApplication->id_currency = $idCurrency;
            $placetApplication->id_order = $idOrder;

            if(!$placetApplication->save()){
                $errorMessage .= sprintf(
                    ' - Failed saving new PlacetApplication object: %s | %s',
                    $this->getVarFormat($idCart, 'id_cart'),
                    $this->getVarFormat($cartTotal, 'cart_total')
                );
                break;
            }

            $result = true;

        } while (1);

        if (!$result){
            // log error
            PrestaShopLogger::addLog(
                $errorMessage, 3, null, "Cart",
                $idCart, true);
        }

        return $result;

    }

    public function getVarFormat($someVar, $varName )
    {
        return sprintf('%s - %s', $varName, print_r($someVar, 1));
    }

    public function handleApplication($token)
    {
        // check state in Placet system
        $applicationStatus = $this->getRequestStatusByToken($token);
        if(empty($applicationStatus)){
            return false; // maybe remove false;
        }


        // lock page
        # this block prevents multiple simultaneous executions by several return requests received from Placet service
        $filename = 'locked';
        $ds = DIRECTORY_SEPARATOR;
        $fullName = _PS_MODULE_DIR_.$ds.$this->name.$ds.$filename;
        $file = fopen($fullName, "w");
        flock($file, LOCK_EX);

        $placetApplication = PlacetApplication::getByToken($token);
        if(empty($placetApplication)){
            return false; // maybe remove false;
        }

        // update status of PlacetApplication
        $placetApplication->updateStatus($applicationStatus);

        // create order if status ok / skip if exists already
        $order = $this->getOrder($placetApplication);

        if(empty($order) ||!Validate::isLoadedObject($order) ){
            return false; // maybe remove false;
        }

        // update order_id of PlacetApplication
        $placetApplication->id_order = $order->id;
        $placetApplication->save();

        // update order status
        $this->updateOrderStatus($order, $placetApplication->application_status);

        return $order;
    }

    public function getRequestStatusByToken($token)
    {
        if (empty($token)){
            // log error
            $errorMessage = 'Executed PlacetGroup::getRequestStatusByToken($token) with empty token';
            PrestaShopLogger::addLog($errorMessage, 3, null, null, null, true);
            return false;
        }

        $url = sprintf(
            '%s/%s/%s/requestStatusByToken',
            Configuration::get('placetgroup'),
            Configuration::get('placetgroup_partneralias'),
            $token
        );

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
//            CURLOPT_CUSTOMREQUEST => 'POST',
//            CURLOPT_POSTFIELDS =>json_encode($params),
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/json',
            ),
        ));

        $response = curl_exec($curl);
        // echo '<pre>';
        // print_r($response);
        // echo '</pre>';
        // die();

        curl_close($curl);
        $response = json_decode($response);

        if(property_exists($response,'request_status')) {
            if(property_exists($response->request_status,'errors')){
                $errorMessage = 'getRequestStatusByToken resulted in error: '
                    .json_encode($response->request_status->errors);
                PrestaShopLogger::addLog($errorMessage, 3, null, null, null, true);
                return false;
            }

            $status = '';

            if(property_exists($response->request_status,'application_status')
                && !empty($response->request_status->application_status)){
                $status = $response->request_status->application_status;
            }
            if(property_exists($response->request_status,'status')
                && !empty($response->request_status->status) && $response->request_status->status === 'declined'){
                $status = $response->request_status->status;
            }

            if($status){
                return $status;
            }
        }

        $errorMessage = 'Invalid response for requestStatusByToken: '.json_encode($response);
        PrestaShopLogger::addLog($errorMessage, 3, null, null, null, true);
        return false;
    }

    public function getOrder($placetApplication)
    {
        if(!Validate::isLoadedObject($placetApplication)){
            $errorMessage = sprintf('Invalid PlacetApplication');
            PrestaShopLogger::addLog($errorMessage, 3, null, null, null, true);
            return false;
        }

        $cart = new Cart((int) $placetApplication->id_cart);

        if(!Validate::isLoadedObject($cart)){
            $errorMessage = sprintf('Failed loading Cart for PlacetApplication with Id %s', $placetApplication->id);
            PrestaShopLogger::addLog($errorMessage, 3, null, null, null, true);
            return false;
        }

        $cartOrderId = Order::getOrderByCartId($cart->id);

        if (!empty($cartOrderId)) {
            $order = new Order($cartOrderId);
            if(!Validate::isLoadedObject($order)){
                $errorMessage = sprintf('Failed loading Order for Cart %s', $cart->id);
                PrestaShopLogger::addLog($errorMessage, 3, null, null, null, true);
                return false;
            }

            return $order;
        }
        return false;
    }

    public function updateOrderStatus($order, $status)
    {
        if (Configuration::get('PS_OS_ERROR') == $order->current_state) {
            // dont update orders with errors
            return;
        }

        if ( $this->orderStateExistsInHistory($order->id, Configuration::get('PS_OS_PAYMENT'))) {
            return;
        }

        if(!isset(PlacetApplication::$statusMap[($status)]['ps_status'])){
            return;
        }

        $newStateName = PlacetApplication::$statusMap[($status)]['ps_status'];
        $newState = Configuration::get($newStateName);

        if($newState === $order->current_state){
            return;
        }

        $order->setCurrentState($newState);

        // new status was 'paid' - modify 'OrderPayment's
        if ('PS_OS_PAYMENT' === $newStateName){
            $this->setPaymentsDescription($order);
        }

    }

    public function orderStateExistsInHistory($orderId, $orderStateId)
    {

        $sql = new DbQuery();
        $sql->select('id_order_state')
            ->from('order_history')
            ->where('id_order = ' . (int) $orderId)
            ->where('id_order_state = ' . (int) $orderStateId);
        $result = Db::getInstance()->executeS($sql);
        foreach ($result as $row) {
            if ((int) $row['id_order_state'] == $orderStateId) return true;
        }
        return false;
    }

    public function setPaymentsDescription($order)
    {
        if (!Validate::isLoadedObject($order)){
            return;
        }

        $orderPayments = $order->getOrderPaymentCollection();

        foreach ($orderPayments as $orderPayment){
            if ($orderPayment->payment_method !== $order->payment){
                $orderPayment->payment_method = $order->payment;
                $orderPayment->save();
            }
        }
    }

    // Legacy redirect link
    // public function getOrderConfRedirectLink($order)
    // {
    //     $url = 'index.php?controller=order-confirmation&id_cart='.$order->id_cart
    //         .'&id_module='.$this->id.'&id_order='.$order->id.'&key='.$order->secure_key;
    //     return $url;
    // }

    public function getOrderConfRedirectLink($order){
        // Check if order was rejected or has errors
        $rejectedState = Configuration::get('PS_OS_PLACETGROUP_REJECTED');
        $errorState = Configuration::get('PS_OS_ERROR');
        $cancelledState = Configuration::get('PS_OS_CANCELED');
        
        if (in_array($order->current_state, [$rejectedState, $errorState, $cancelledState])) {
            // Redirect to order history with error message for rejected payments
            return 'index.php?controller=history&placet_error=declined';
        }
        
        // Normal success redirect
        $url = 'index.php?controller=order-confirmation&id_cart='.$order->id_cart
            .'&id_module='.$this->id.'&id_order='.$order->id.'&key='.$order->secure_key;
        return $url;
    }


    public function writeLog ($logObj, $identifier = '', $includeMetaData = true)
    {
        $dbt=debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);
        $method = isset($dbt[1]['function']) ? $dbt[1]['function'] : '';
        $line = isset($dbt[0]['line']) ? ':'.$dbt[0]['line'] : '';

        if ($includeMetaData) {
            $log = date('d-m-Y H:i:s').' - '.$identifier.' - '.$method.$line;
        } else {
            $log = $identifier;
        }

        $log .= PHP_EOL;
        if (!empty($logObj)) {
            $logText = print_r( $logObj, true);
            $log .= $logText;
            $log .= PHP_EOL;
        }

        $ds = DIRECTORY_SEPARATOR;
        $path = _PS_MODULE_DIR_.$ds.$this->name.$ds;
        $fileName = date('Ymd').'.log';
        $file = $path.$fileName;

        file_put_contents($file, $log, FILE_APPEND | LOCK_EX);
    }

    public function createOrderStates()
    {
        $states = [
            [
                'status_name' => 'PS_OS_PLACETGROUP_AWAITING',
                'description' => sprintf('Awaiting PlacetGroup confirmation'),
                'descriptionEE' => sprintf('Ootab PlacetGroup kinnitust'),
                'color' => "#4169E1"
            ],
            [
                'status_name' => 'PS_OS_PLACETGROUP_REJECTED',
                'description' => sprintf('PlacetGroup payment rejected'),
                'descriptionEE' => sprintf('PlacetGroup taotlus lükati tagasi'),
                'color' => "#DC143C"
            ]
        ];

        $result = true;
        foreach ($states as $state) {
            if(!$this->addOrderState($state)) {
                $result = false;
            }
        }

        return $result;
    }

    /**
     * Create Order State
     * @return boolean  true, if successful
     */
    protected function addOrderState($stateArr)
    {
        $orderStateExist = false;
        $status_name = $stateArr['status_name'];
        $orderStateId = Configuration::get($status_name);
        $description = $stateArr['description'];
        $descriptionEE = $stateArr['descriptionEE'];
        if ($orderStateId) {
            $orderState = new OrderState($orderStateId);
            if ($orderState->id && !$orderState->deleted) {
                $orderStateExist = true;
            }
        } else {
            $query = 'SELECT os.`id_order_state` '.
                'FROM `%1$sorder_state_lang` osl '.
                'LEFT JOIN `%1$sorder_state` os '.
                'ON osl.`id_order_state`=os.`id_order_state` '.
                'WHERE osl.`name`="%2$s" OR osl.`name`="%3$s" AND os.`deleted`=0';

            $orderStateId =  Db::getInstance()->getValue(sprintf($query, _DB_PREFIX_, $description, $descriptionEE));
            if ($orderStateId) {
                Configuration::updateValue($status_name, $orderStateId);
                $orderStateExist = true;
            }
        }

        if (!$orderStateExist) {

            $languages = Language::getLanguages(false);
            $orderState = new OrderState();
            $estLangID = Language::getIdByIso('et');

            foreach ($languages as $lang) {
                if ($lang['id_lang'] == $estLangID) {
                    $orderState->name[$lang['id_lang']] = $descriptionEE;
                } else {
                    $orderState->name[$lang['id_lang']] = $description;
                }
            }
            $orderState->send_email = 0;
            $orderState->invoice = 0;
            $orderState->color = $stateArr['color'];
            $orderState->unremovable = 0;
            $orderState->logable = 0;
            $orderState->delivery = 0;
            $orderState->hidden = 0;
            if ($orderState->add()) {
                Configuration::updateValue($status_name, $orderState->id);
                $orderStateExist = true;
            }
        }

        return ($orderStateExist);
    }
}
