<?php
/**
 * Placet Group Payment Gateway Catalog Controller for OpenCart 3
 */
class ControllerExtensionPaymentPlacetgroup extends Controller {
    /**
     * Index - Display payment method
     */
    public function index() {
        $this->load->language('extension/payment/placetgroup');

        $data['button_confirm'] = $this->language->get('button_confirm');
        $data['text_loading'] = $this->language->get('text_wait');
        
        return $this->load->view('extension/payment/placetgroup', $data);
    }

    /**
     * Confirm - Process payment and redirect to Placet Group
     */
    public function confirm() {
        $this->load->language('extension/payment/placetgroup');

        $json = array();

        // Validate session
        if (!isset($this->session->data['order_id'])) {
            $json['error'] = $this->language->get('error_order');
        }

        // Validate order
        if (!$json) {
            $this->load->model('checkout/order');

            $order_info = $this->model_checkout_order->getOrder($this->session->data['order_id']);

            if (!$order_info) {
                $json['error'] = $this->language->get('error_order');
            }
        }

        // Validate payment method
        if (!isset($this->session->data['payment_method']['code']) || 
            $this->session->data['payment_method']['code'] != 'placetgroup') {
            $json['error'] = $this->language->get('error_payment_method');
        }

        // Validate currency (EUR only)
        if (!$json) {
            if ($this->session->data['currency'] != 'EUR') {
                $json['error'] = $this->language->get('error_currency');
            }
        }

        if (!$json) {
            // Load Placet Group library
            require_once(DIR_SYSTEM . 'library/placetgroup.php');
            $placetgroup = new Placetgroup($this->registry);

            // Update order status to "Awaiting Placet Group Confirmation"
            $awaitingStatusId = (int)$this->config->get('payment_placetgroup_awaiting_status_id');
            
            if ($awaitingStatusId > 0) {
                $this->model_checkout_order->addOrderHistory(
                    $this->session->data['order_id'],
                    $awaitingStatusId,
                    'Awaiting Placet Group payment confirmation',
                    false
                );
            }

            // Prepare order items
            $this->load->model('account/order');
            $products = $this->model_account_order->getOrderProducts($this->session->data['order_id']);
            
            $items = array();
            foreach ($products as $product) {
                $items[] = array('name' => $product['name']);
            }

            // Get currency info
            $this->load->model('localisation/currency');
            $currency_info = $this->model_localisation_currency->getCurrencyByCode($this->session->data['currency']);

            // Prepare callback URLs
            if ($this->request->server['HTTPS']) {
                $callbackUrl = HTTPS_SERVER . 'index.php?route=extension/payment/placetgroup/callback';
            } else {
                $callbackUrl = HTTP_SERVER . 'index.php?route=extension/payment/placetgroup/callback';
            }
            
            $successUrl = str_replace('%rt%', '%rt%', $callbackUrl . '&status=success&token=%rt%');
            $errorUrl = str_replace('%rt%', '%rt%', $callbackUrl . '&status=error&token=%rt%');

            // Request payment token
            $tokenData = $placetgroup->requestToken(array(
                'amount' => $order_info['total'],
                'locale' => 'et',
                'items' => $items,
                'customer_id' => $order_info['customer_id'],
                'order_id' => $order_info['order_id'],
                'success_url' => $successUrl,
                'error_url' => $errorUrl,
                'notification_url' => $callbackUrl
            ));

            if ($tokenData === false || empty($tokenData['token'])) {
                $json['error'] = $this->language->get('error_payment_init');
            } else {
                // Create application record
                $placetgroup->createApplication(array(
                    'token' => $tokenData['token'],
                    'order_id' => $order_info['order_id'],
                    'amount' => $order_info['total'],
                    'currency_id' => $currency_info['currency_id'],
                    'currency_code' => $this->session->data['currency']
                ));

                // Redirect to Placet Group
                if (!empty($tokenData['redirect_url'])) {
                    $json['redirect'] = $tokenData['redirect_url'];
                } else {
                    // Fallback: construct URL manually
                    $baseUrl = rtrim($this->config->get('payment_placetgroup_url'), '/');
                    $alias = trim($this->config->get('payment_placetgroup_partner_alias'), '/');
                    $json['redirect'] = $baseUrl . '/' . $alias . '/' . $tokenData['token'];
                }
            }
        }

        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($json));
    }

    /**
     * Callback - Handle return and notification from Placet Group
     */
    public function callback() {
        // Get token from various sources
        $token = null;

        // Check POST data (JSON notification)
        $rawInput = file_get_contents('php://input');
        if (!empty($rawInput)) {
            $json = json_decode($rawInput);
            if (isset($json->notification->token)) {
                $token = $json->notification->token;
            }
        }

        // Check GET parameter
        if (empty($token) && isset($this->request->get['token'])) {
            $token = $this->request->get['token'];
        }

        // Check for requestToken parameter (alternative)
        if (empty($token) && isset($this->request->get['requestToken'])) {
            $token = $this->request->get['requestToken'];
        }

        if (empty($token)) {
            $this->log->write('PLACETGROUP ERROR: Callback received without token');
            
            // Redirect to checkout on error
            $this->response->redirect($this->url->link('checkout/failure', '', true));
            return;
        }

        // Load Placet Group library
        require_once(DIR_SYSTEM . 'library/placetgroup.php');
        $placetgroup = new Placetgroup($this->registry);

        // Handle the callback
        $orderInfo = $placetgroup->handleCallback($token);

        if ($orderInfo === false) {
            $this->log->write('PLACETGROUP ERROR: Failed to handle callback for token: ' . $token);
            
            // For API notifications, just return success
            if (!empty($rawInput)) {
                echo 'success';
                exit;
            }

            // For user redirects, go to failure page
            $this->response->redirect($this->url->link('checkout/failure', '', true));
            return;
        }

        // For API notifications, just return success
        if (!empty($rawInput)) {
            echo 'success';
            exit;
        }

        // For user redirects, check status and redirect accordingly
        $application = $placetgroup->getApplicationByToken($token);
        
        if ($application && $application['application_status'] === 'declined') {
            // Redirect to failure page for declined payments
            $this->response->redirect($this->url->link('checkout/failure', '', true));
        } else {
            // Redirect to success page
            $this->response->redirect($this->url->link('checkout/success', '', true));
        }
    }
}

