<?php
/**
 * Placet Group Payment Gateway Library for OpenCart 3
 * 
 * Handles API communication and business logic for Placet Group payment processing
 */
class Placetgroup {
    private $registry;
    private $db;
    private $config;
    private $log;

    /**
     * Status mapping between Placet Group and OpenCart
     * Maps Placet Group payment statuses to OpenCart config keys
     */
    const STATUS_MAP = array(
        'new' => 'awaiting',           // Payment initiated but not yet processed
        'pending' => 'awaiting',       // Payment pending
        'approved' => 'payment',       // Payment approved
        'confirmed' => 'payment',      // Payment confirmed (alternative to approved)
        'completed' => 'payment',      // Payment completed
        'declined' => 'rejected',      // Payment declined
        'rejected' => 'rejected',      // Payment rejected
        'cancelled' => 'rejected',     // Payment cancelled
        'failed' => 'rejected'         // Payment failed
    );

    /**
     * Constructor
     *
     * @param object $registry Registry object
     */
    public function __construct($registry) {
        $this->registry = $registry;
        $this->db = $registry->get('db');
        $this->config = $registry->get('config');
        $this->log = $registry->get('log');
    }

    /**
     * Request payment token from Placet Group API
     *
     * @param array $data Payment request data
     * @return array|false Token data or false on failure
     */
    public function requestToken($data) {
        $url = rtrim($this->config->get('payment_placetgroup_url'), '/');
        $alias = trim($this->config->get('payment_placetgroup_partner_alias'), '/');
        
        if (empty($url) || empty($alias)) {
            $this->log->write('PLACETGROUP ERROR: Missing configuration (URL or Partner Alias)');
            return false;
        }

        $endpoint = $url . '/' . $alias . '/requestToken';

        $params = array(
            'request_token' => array(
                'amount' => (float)$data['amount'],
                'locale' => isset($data['locale']) ? $data['locale'] : 'et',
                'product_items' => isset($data['items']) ? $data['items'] : array(),
                'client_id' => (string)$data['customer_id'],
                'client_ip' => $this->getClientIp(),
                'success_url' => $data['success_url'],
                'error_url' => $data['error_url'],
                'notification_url' => $data['notification_url'],
                'cart_id' => (string)(isset($data['cart_id']) ? $data['cart_id'] : ''),
                'order_id' => (string)$data['order_id']
            )
        );

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $endpoint,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($params),
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/json',
            ),
        ));

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curlError = curl_error($curl);
        curl_close($curl);

        if ($response === false || empty($response) || $httpCode < 200 || $httpCode >= 300) {
            $this->log->write('PLACETGROUP ERROR: requestToken failed - HTTP: ' . $httpCode . ', cURL: ' . $curlError);
            return false;
        }

        $decoded = json_decode($response);

        if (!is_object($decoded) || !isset($decoded->request_token)) {
            $this->log->write('PLACETGROUP ERROR: Invalid JSON response from requestToken: ' . $response);
            return false;
        }

        $token = null;
        $redirectUrl = null;

        if (is_object($decoded->request_token)) {
            $token = isset($decoded->request_token->token) ? (string)$decoded->request_token->token : '';
            $redirectUrl = isset($decoded->request_token->redirect_url) ? (string)$decoded->request_token->redirect_url : '';
        } else {
            $token = (string)$decoded->request_token;
        }

        if (empty($token)) {
            $this->log->write('PLACETGROUP ERROR: No token in response');
            return false;
        }

        return array(
            'token' => $token,
            'redirect_url' => $redirectUrl
        );
    }

    /**
     * Request application status by token
     *
     * @param string $token Payment token
     * @return string|false Status string or false on failure
     */
    public function requestStatusByToken($token) {
        if (empty($token)) {
            $this->log->write('PLACETGROUP ERROR: requestStatusByToken called with empty token');
            return false;
        }

        $url = rtrim($this->config->get('payment_placetgroup_url'), '/');
        $alias = trim($this->config->get('payment_placetgroup_partner_alias'), '/');

        if (empty($url) || empty($alias)) {
            $this->log->write('PLACETGROUP ERROR: Missing configuration for status check');
            return false;
        }

        $endpoint = $url . '/' . $alias . '/' . $token . '/requestStatusByToken';

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $endpoint,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/json',
            ),
        ));

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curlError = curl_error($curl);
        curl_close($curl);

        if ($response === false || empty($response) || $httpCode < 200 || $httpCode >= 300) {
            $this->log->write('PLACETGROUP ERROR: requestStatusByToken failed - HTTP: ' . $httpCode . ', cURL: ' . $curlError);
            return false;
        }

        $decoded = json_decode($response);

        if (!is_object($decoded) || !isset($decoded->request_status)) {
            $this->log->write('PLACETGROUP ERROR: Invalid response from requestStatusByToken: ' . $response);
            return false;
        }

        // Log the full response for debugging
        $this->log->write('PLACETGROUP DEBUG: Full API response: ' . $response);

        if (isset($decoded->request_status->errors)) {
            $this->log->write('PLACETGROUP ERROR: API returned errors: ' . json_encode($decoded->request_status->errors));
            return false;
        }

        $status = '';

        // Check for application_status first (this is the payment status)
        if (isset($decoded->request_status->application_status) && !empty($decoded->request_status->application_status)) {
            $status = $decoded->request_status->application_status;
            $this->log->write('PLACETGROUP DEBUG: Found application_status: ' . $status);
        }

        // Check for status field (might override application_status in some cases)
        if (isset($decoded->request_status->status) && !empty($decoded->request_status->status)) {
            $apiStatus = $decoded->request_status->status;
            $this->log->write('PLACETGROUP DEBUG: Found status field: ' . $apiStatus);
            
            // If status is 'declined', it overrides application_status
            if ($apiStatus === 'declined' || $apiStatus === 'rejected') {
                $status = 'declined';
            }
            // If status is 'approved' or 'confirmed', use it
            elseif ($apiStatus === 'approved' || $apiStatus === 'confirmed' || $apiStatus === 'completed') {
                $status = 'approved';
            }
        }

        if (empty($status)) {
            $this->log->write('PLACETGROUP ERROR: No status in response: ' . $response);
            return false;
        }

        $this->log->write('PLACETGROUP DEBUG: Final determined status: ' . $status);
        return $status;
    }

    /**
     * Create application record in database
     *
     * @param array $data Application data
     * @return int|false Application ID or false on failure
     */
    public function createApplication($data) {
        $token = $this->db->escape(isset($data['token']) ? $data['token'] : '');
        $orderId = (int)(isset($data['order_id']) ? $data['order_id'] : 0);
        $amount = (float)(isset($data['amount']) ? $data['amount'] : 0);
        $currencyId = (int)(isset($data['currency_id']) ? $data['currency_id'] : 0);
        $currencyCode = $this->db->escape(isset($data['currency_code']) ? $data['currency_code'] : '');

        if (empty($token) || $orderId <= 0 || $amount <= 0 || $currencyId <= 0) {
            $this->log->write('PLACETGROUP ERROR: Invalid data for createApplication');
            return false;
        }

        // Check if application with this token already exists
        $checkQuery = $this->db->query("SELECT application_id FROM `" . DB_PREFIX . "placetgroup_application` WHERE token = '" . $token . "'");

        if ($checkQuery->num_rows > 0) {
            $this->log->write('PLACETGROUP ERROR: Application with token ' . $token . ' already exists');
            return false;
        }

        $this->db->query("INSERT INTO `" . DB_PREFIX . "placetgroup_application` SET 
            token = '" . $token . "',
            order_id = '" . $orderId . "',
            amount = '" . $amount . "',
            currency_id = '" . $currencyId . "',
            currency_code = '" . $currencyCode . "',
            application_status = NULL,
            date_added = NOW(),
            date_modified = NOW()
        ");

        return $this->db->getLastId();
    }

    /**
     * Get application by token
     *
     * @param string $token Payment token
     * @return array|false Application data or false if not found
     */
    public function getApplicationByToken($token) {
        $token = $this->db->escape($token);

        $query = $this->db->query("SELECT * FROM `" . DB_PREFIX . "placetgroup_application` WHERE token = '" . $token . "' ORDER BY date_modified DESC LIMIT 1");

        if ($query->num_rows > 0) {
            return $query->row;
        }

        return false;
    }

    /**
     * Update application status
     *
     * @param string $token Payment token
     * @param string $status New status
     * @return bool Success
     */
    public function updateApplicationStatus($token, $status) {
        $token = $this->db->escape($token);
        $status = $this->db->escape($status);

        $this->db->query("UPDATE `" . DB_PREFIX . "placetgroup_application` SET 
            application_status = '" . $status . "',
            date_modified = NOW()
            WHERE token = '" . $token . "'
        ");

        return true;
    }

    /**
     * Handle payment callback/notification
     *
     * @param string $token Payment token
     * @return array|false Order info or false on failure
     */
    public function handleCallback($token) {
        // Get application status from API
        $status = $this->requestStatusByToken($token);

        if ($status === false) {
            $this->log->write('PLACETGROUP ERROR: Failed to get status for token: ' . $token);
            return false;
        }

        // Get application record
        $application = $this->getApplicationByToken($token);

        if ($application === false) {
            $this->log->write('PLACETGROUP ERROR: Application not found for token: ' . $token);
            return false;
        }

        // Update application status
        $this->updateApplicationStatus($token, $status);

        // Load order model
        $this->registry->get('load')->model('checkout/order');
        $modelCheckoutOrder = $this->registry->get('model_checkout_order');

        $orderInfo = $modelCheckoutOrder->getOrder($application['order_id']);

        if (empty($orderInfo)) {
            $this->log->write('PLACETGROUP ERROR: Order not found: ' . $application['order_id']);
            return false;
        }

        // Store the Placet token as transaction ID in OpenCart order
        $this->db->query("UPDATE `" . DB_PREFIX . "order` SET payment_code = '" . $this->db->escape($token) . "' WHERE order_id = '" . (int)$application['order_id'] . "'");
        
        $this->log->write('PLACETGROUP: Transaction ID saved for order #' . $application['order_id'] . ': ' . $token);

        // Update order status based on payment status (this adds the status to order history)
        $this->updateOrderStatus($application['order_id'], $status, $token);

        return $orderInfo;
    }

    /**
     * Update order status based on payment status
     *
     * @param int $orderId Order ID
     * @param string $status Payment status
     * @param string $token Payment token (transaction ID)
     * @return void
     */
    private function updateOrderStatus($orderId, $status, $token = '') {
        $statusMap = self::STATUS_MAP;
        
        $this->log->write('PLACETGROUP: Updating order #' . $orderId . ' with payment status: ' . $status);
        
        if (!isset($statusMap[$status])) {
            $this->log->write('PLACETGROUP WARNING: Unknown status "' . $status . '" for order #' . $orderId);
            return;
        }

        $orderStatusKey = $statusMap[$status];

        if ($orderStatusKey === null) {
            // Status 'new' - no action needed
            $this->log->write('PLACETGROUP: Order #' . $orderId . ' status is "new", no action taken');
            return;
        }

        $configKey = 'payment_placetgroup_' . $orderStatusKey . '_status_id';
        $orderStatusId = (int)$this->config->get($configKey);

        $this->log->write('PLACETGROUP: Looking for config key: ' . $configKey . ' = ' . $orderStatusId);

        if ($orderStatusId <= 0) {
            $this->log->write('PLACETGROUP ERROR: Order status ID not configured for key: ' . $configKey . ' (status: ' . $status . ')');
            return;
        }

        // Load order model
        $this->registry->get('load')->model('checkout/order');
        $modelCheckoutOrder = $this->registry->get('model_checkout_order');

        // Build comment with transaction ID
        $comment = 'Placet Group payment status: ' . $status;
        if (!empty($token)) {
            $comment .= "\nTransaction ID: " . $token;
        }
        
        $notify = ($status === 'approved');

        $modelCheckoutOrder->addOrderHistory($orderId, $orderStatusId, $comment, $notify);
        
        $this->log->write('PLACETGROUP: Order #' . $orderId . ' status updated to order_status_id: ' . $orderStatusId . ' (payment status: ' . $status . ', transaction: ' . $token . ')');
    }

    /**
     * Get client IP address
     *
     * @return string IP address
     */
    private function getClientIp() {
        $keys = array(
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        );

        foreach ($keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ips = explode(',', $_SERVER[$key]);
                foreach ($ips as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                        return $ip;
                    }
                }
            }
        }

        // Fallback to any valid IP
        foreach ($keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ips = explode(',', $_SERVER[$key]);
                foreach ($ips as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP)) {
                        return $ip;
                    }
                }
            }
        }

        return '127.0.0.1';
    }
}

