<?php
/**
 * WC_Placet_Group_Blocks_Support class
 *
 * @package WC_Placet_Group
 */

defined( 'ABSPATH' ) || exit;

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

/**
 * Placet Group Blocks Payment Method
 *
 * @since 2.0.0
 */
final class WC_Placet_Group_Blocks_Support extends AbstractPaymentMethodType {

    /**
     * Payment method name/id/slug.
     *
     * @var string
     */
    protected $name = 'placet_group';

    /**
     * Gateway instance
     *
     * @var WC_Payment_Gateway
     */
    private $gateway;

    /**
     * Initializes the payment method type.
     */
    public function initialize() {
        $this->settings = get_option( 'woocommerce_placet_group_settings', array() );
        
        // Get gateway instance
        $gateways = WC()->payment_gateways->payment_gateways();
        $this->gateway = isset( $gateways[ $this->name ] ) ? $gateways[ $this->name ] : null;
    }

    /**
     * Returns if this payment method should be active. If false, the scripts will not be enqueued.
     *
     * @return boolean
     */
    public function is_active() {
        // Check if gateway is enabled in settings
        return ! empty( $this->settings['enabled'] ) && 'yes' === $this->settings['enabled'];
    }

    /**
     * Returns an array of scripts/handles to be registered for this payment method.
     *
     * @return array
     */
    public function get_payment_method_script_handles() {
        $script_asset_path = WC_PLACET_GROUP_PLUGIN_DIR . 'assets/js/blocks.asset.php';
        $script_asset      = file_exists( $script_asset_path )
            ? require $script_asset_path
            : array(
                'dependencies' => array(
                    'wc-blocks-registry',
                    'wc-settings',
                    'wp-element',
                    'wp-html-entities',
                    'wp-i18n',
                ),
                'version'      => WC_PLACET_GROUP_VERSION,
            );
        
        $script_url = WC_PLACET_GROUP_PLUGIN_URL . 'assets/js/blocks.js';

        wp_register_script(
            'wc-placet-group-blocks',
            $script_url,
            $script_asset['dependencies'],
            $script_asset['version'],
            true
        );

        // Set script translations if available
        if ( function_exists( 'wp_set_script_translations' ) ) {
            wp_set_script_translations( 'wc-placet-group-blocks', 'wc-placet-group' );
        }

        return array( 'wc-placet-group-blocks' );
    }

    /**
     * Returns an array of key=>value pairs of data made available to the payment methods script.
     *
     * @return array
     */
    public function get_payment_method_data() {
        if ( ! $this->gateway ) {
            // Return minimal data if gateway not available
            return array(
                'title'       => __( 'Placet Järelmaks', 'wc-placet-group' ),
                'description' => '',
                'supports'    => array(),
                'icon'        => WC_PLACET_GROUP_PLUGIN_URL . 'assets/images/placet-jarelmaks-kompakt.png',
            );
        }

        // Get the icon URL directly from the gateway property
        $icon_url = ! empty( $this->gateway->icon ) ? $this->gateway->icon : WC_PLACET_GROUP_PLUGIN_URL . 'assets/images/placet-jarelmaks-kompakt.png';

        $data = array(
            'title'       => $this->gateway->get_title(),
            'description' => $this->gateway->get_description(),
            'supports'    => $this->gateway->supports,
            'icon'        => $icon_url,
        );

        // Debug logging (remove in production)
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'Placet Group Blocks Data: ' . print_r( $data, true ) );
        }

        return $data;
    }
}

