<?php
/**
 * WC_Placet_Group_Gateway class
 *
 * @package WC_Placet_Group
 */

defined( 'ABSPATH' ) || exit;

/**
 * Placet Group Payment Gateway
 *
 * Provides a Placet Järelmaks Payment Gateway with HPOS and Block Checkout support
 *
 * @class       WC_Placet_Group_Gateway
 * @extends     WC_Payment_Gateway
 * @version     2.0.0
 * @package     WooCommerce/Classes/Payment
 */
class WC_Placet_Group_Gateway extends WC_Payment_Gateway {

    /**
     * API URL
     *
     * @var string
     */
    private $api_url;

    /**
     * Partner ID
     *
     * @var string
     */
    private $partner_id;

    /**
     * Test mode
     *
     * @var bool
     */
    private $test_mode;

    /**
     * Constructor for the gateway
     */
    public function __construct() {
        $this->id                 = 'placet_group';
        $this->icon               = apply_filters(
            'woocommerce_placet_group_icon',
            WC_PLACET_GROUP_PLUGIN_URL . 'assets/images/placet-jarelmaks-kompakt.png'
        );
        $this->has_fields         = false;
        $this->method_title       = __( 'Placet Järelmaks', 'wc-placet-group' );
        $this->method_description = __( 'Accept installment payments through Placet Järelmaks payment gateway.', 'wc-placet-group' );

        // Supports
        $this->supports = array(
            'products',
        );

        // Load the settings FIRST
        $this->init_form_fields();
        $this->init_settings();

        // THEN access settings (after init_settings has loaded them)
        $this->title              = $this->get_option( 'title' );
        $this->description        = $this->get_option( 'description' );
        $this->partner_id         = $this->get_option( 'partner_id' );
        $this->test_mode          = 'yes' === $this->get_option( 'test_mode', 'no' );
        $this->enabled            = $this->get_option( 'enabled' );

        // Build API URL based on test mode and partner ID
        $this->api_url = $this->build_api_url();

        // Actions
        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
        //add_action( 'woocommerce_api_wc_placet_group', array( $this, 'handle_webhook' ) );
        add_action( 'woocommerce_thankyou_' . $this->id, array( $this, 'thankyou_page' ) );

        // Enqueue scripts
        add_action( 'wp_enqueue_scripts', array( $this, 'payment_scripts' ) );

        // Handle payment error messages
        add_action( 'template_redirect', array( $this, 'handle_payment_error_message' ) );
    }

    /**
     * Initialize Gateway Settings Form Fields
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled'     => array(
                'title'       => __( 'Enable/Disable', 'wc-placet-group' ),
                'label'       => __( 'Enable Placet Järelmaks', 'wc-placet-group' ),
                'type'        => 'checkbox',
                'description' => '',
                'default'     => 'no',
            ),
            'title'       => array(
                'title'       => __( 'Title', 'wc-placet-group' ),
                'type'        => 'text',
                'description' => __( 'This controls the title which the user sees during checkout.', 'wc-placet-group' ),
                'default'     => __( 'Placet Järelmaks', 'wc-placet-group' ),
                'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __( 'Description', 'wc-placet-group' ),
                'type'        => 'textarea',
                'description' => __( 'Payment method description that the customer will see on your checkout.', 'wc-placet-group' ),
                'default'     => __( 'Maksa osamaksete kaupa soodsalt Placet Järelmaksuga', 'wc-placet-group' ),
                'desc_tip'    => true,
            ),
            'partner_id'  => array(
                'title'       => __( 'Partner ID', 'wc-placet-group' ),
                'type'        => 'text',
                'description' => __( 'Enter your Partner ID provided by Placet Group.', 'wc-placet-group' ),
                'default'     => '',
                'desc_tip'    => true,
            ),
            'test_mode'   => array(
                'title'       => __( 'Test Mode', 'wc-placet-group' ),
                'label'       => __( 'Enable Test Mode', 'wc-placet-group' ),
                'type'        => 'checkbox',
                'description' => __( 'Enable this to use the test environment (https://web-partner.placet.ee). Disable for production environment (https://online.placet.ee).', 'wc-placet-group' ),
                'default'     => '',
                'desc_tip'    => true,
            ),
        );
    }

    /**
     * Build API URL based on partner ID and test mode
     *
     * @return string
     */
    private function build_api_url() {
        if ( empty( $this->partner_id ) ) {
            return '';
        }

        $base_url = $this->test_mode 
            ? 'https://web-partner.placet.ee/' 
            : 'https://online.placet.ee/';

        return $base_url . $this->partner_id . '/requestToken';
    }

    /**
     * Payment scripts
     */
    public function payment_scripts() {
        // Only enqueue on checkout page
        if ( ! is_checkout() ) {
            return;
        }

        wp_enqueue_style(
            'wc-placet-group-checkout',
            WC_PLACET_GROUP_PLUGIN_URL . 'assets/css/checkout.css',
            array(),
            WC_PLACET_GROUP_VERSION
        );
    }

    /**
     * Process the payment and return the result
     *
     * @param int $order_id Order ID.
     * @return array
     */
    public function process_payment( $order_id ) {
        $order = wc_get_order( $order_id );

        if ( ! $order ) {
            wc_add_notice( __( 'Order not found.', 'wc-placet-group' ), 'error' );
            return array(
                'result'   => 'failure',
                'redirect' => '',
            );
        }

        // Check if order needs payment
        if ( ! $order->needs_payment() ) {
            return array(
                'result'   => 'success',
                'redirect' => $this->get_return_url( $order ),
            );
        }

        // Validate API URL and Partner ID
        if ( empty( $this->partner_id ) || empty( $this->api_url ) ) {
            wc_add_notice( __( 'Payment gateway is not configured properly. Partner ID is missing. Please contact the site administrator.', 'wc-placet-group' ), 'error' );
            return array(
                'result'   => 'failure',
                'redirect' => '',
            );
        }

        try {
            // Check if we already have a redirect URL in transient (prevent double processing)
            $transient_key = 'placet_group_redirect_' . $order_id;
            $redirect_url = get_transient( $transient_key );

            if ( $redirect_url ) {
                // Clean up transient after use
                delete_transient( $transient_key );
                
                return array(
                    'result'   => 'success',
                    'redirect' => $redirect_url,
                );
            }

            // Prepare payment request
            $payment_data = $this->prepare_payment_data( $order );

            // Send request to Placet API
            $response = $this->send_payment_request( $payment_data );

            if ( is_wp_error( $response ) ) {
                throw new Exception( $response->get_error_message() );
            }

            // Validate response
            $redirect_url = $this->validate_payment_response( $response, $order );

            if ( ! $redirect_url ) {
                throw new Exception( __( 'Invalid payment response from gateway.', 'wc-placet-group' ) );
            }

            // Store the redirect URL in transient for 5 minutes
            set_transient( $transient_key, $redirect_url, 300 );

            // Add order note
            $order->add_order_note(
                sprintf(
                    __( 'Customer redirected to Placet payment gateway. Transaction initiated.', 'wc-placet-group' )
                )
            );

            // Return success and redirect to payment gateway
            return array(
                'result'   => 'success',
                'redirect' => $redirect_url,
            );

        } catch ( Exception $e ) {
            wc_add_notice( $e->getMessage(), 'error' );
            
            // Log error
            // $this->log( 'Payment processing error: ' . $e->getMessage() );
            
            // Add order note
            $order->add_order_note(
                sprintf(
                    __( 'Payment error: %s', 'wc-placet-group' ),
                    $e->getMessage()
                )
            );

            return array(
                'result'   => 'failure',
                'redirect' => '',
            );
        }
    }

    /**
     * Prepare payment data for API request
     *
     * @param WC_Order $order Order object.
     * @return array
     */
    private function prepare_payment_data( $order ) {
        // Get order items
        $items = array();
        foreach ( $order->get_items() as $item ) {
            $items[] = array(
                'name' => $item->get_name(),
            );
        }

        // Get customer data
        $user_id = $order->get_user_id();
        if ( ! $user_id && is_user_logged_in() ) {
            $user_id = get_current_user_id();
        }

        // Prepare checkout URLs
        $checkout_url = $order->get_checkout_payment_url( false );
        
        // Build payment data
        $payment_data = array(
            'request_token' => array(
                'amount'        => (float) $order->get_total(),
                'locale'        => 'et', //$this->get_locale(),
                'product_items' => $items,
                'client_id'     => (string) $user_id,
                'client_ip'     => $this->get_client_ip(),
                'success_url'   => $order->get_checkout_order_received_url(),
                'error_url'     => add_query_arg( 'placet_error', '1', $checkout_url ),
                'order_id'      => $order->get_id(),
                'cart_id'       => $order->get_id(),
            ),
        );

        return apply_filters( 'wc_placet_group_payment_data', $payment_data, $order );
    }


    /**
     * Send payment request to Placet API
     *
     * @param array $payment_data Payment data.
     * @return array|WP_Error
     */
    private function send_payment_request( $payment_data ) {
        $response = wp_remote_post(
            $this->api_url,
            array(
                'headers'     => array(
                    'Content-Type' => 'application/json',
                    'Accept'       => 'application/json',
                ),
                'body'        => wp_json_encode( $payment_data ),
                'method'      => 'POST',
                'timeout'     => 30,
                'data_format' => 'body',
            )
        );

        if ( is_wp_error( $response ) ) {
            // $this->log( 'API Request Error: ' . $response->get_error_message() );
            return $response;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body );

        // $this->log( 'API Response: ' . print_r( $data, true ) );

        return $data;
    }

    /**
     * Validate payment response from API
     *
     * @param object $response API response.
     * @param WC_Order $order Order object.
     * @return string|false Redirect URL or false on failure.
     */
    private function validate_payment_response( $response, $order ) {
        if ( ! is_object( $response ) || ! property_exists( $response, 'request_token' ) ) {
            // $this->log( 'Invalid response structure' );
            return false;
        }

        // $this->log( 'Validate payment response: ' . print_r( $response, true ), 'debug' );

        $request_token = $response->request_token;

        // Check for errors
        if ( property_exists( $request_token, 'errors' ) && ! empty( $request_token->errors ) ) {
            $error_message = is_array( $request_token->errors ) && isset( $request_token->errors['summary'] )
                ? $request_token->errors['summary']
                : __( 'Payment gateway error occurred.', 'wc-placet-group' );
            
            throw new Exception( esc_html( $error_message ) );
        }

        // Save transaction ID if provided
        if ( property_exists( $request_token, 'token' ) && ! empty( $request_token->token ) ) {
            $transaction_id = sanitize_text_field( $request_token->token );
            
            // Save transaction ID using WooCommerce's built-in method
            $order->set_transaction_id( $transaction_id );
            
            // Also save as meta for reference
            $order->update_meta_data( '_placet_transaction_id', $transaction_id );
            $order->save();
            
            // Add order note with transaction ID
            $order->add_order_note(
                sprintf(
                    __( 'Placet transaction initiated. Transaction ID: %s', 'wc-placet-group' ),
                    $transaction_id
                )
            );
            
            // $this->log( 'Transaction ID saved: ' . $transaction_id );
        }

        // Get redirect URL
        if ( property_exists( $request_token, 'redirect_url' ) && ! empty( $request_token->redirect_url ) ) {
            return esc_url_raw( $request_token->redirect_url );
        }

        return false;
    }

    /**
     * Get client IP address
     *
     * @return string
     */
    private function get_client_ip() {
        $ip_keys = array(
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR',
        );

        foreach ( $ip_keys as $key ) {
            if ( ! empty( $_SERVER[ $key ] ) ) {
                $ip = sanitize_text_field( wp_unslash( $_SERVER[ $key ] ) );
                
                // Handle comma-separated IPs (proxy chains)
                if ( strpos( $ip, ',' ) !== false ) {
                    $ips = explode( ',', $ip );
                    $ip  = trim( $ips[0] );
                }

                // Validate IP
                if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
                    return $ip;
                }
            }
        }

        return 'UNKNOWN';
    }

    /**
     * Get locale for payment gateway
     *
     * @return string
     */
    private function get_locale() {
        $locale = get_locale();
        
        // Convert WordPress locale to simple language code
        if ( strpos( $locale, 'et_' ) === 0 ) {
            return 'et';
        } elseif ( strpos( $locale, 'en_' ) === 0 ) {
            return 'en';
        }

        // Default to Estonian
        return 'et';
    }

    // /**
    //  * Handle webhook/callback from payment gateway
    //  */
    // public function handle_webhook() {
    //     $this->log( 'Webhook received: ' . print_r( $_REQUEST, true ) );

    //     // Get raw POST data
    //     $raw_post = file_get_contents( 'php://input' );
    //     $this->log( 'Webhook raw POST: ' . $raw_post );

    //     // Parse JSON if available
    //     $webhook_data = json_decode( $raw_post );
        
    //     if ( ! $webhook_data ) {
    //         // Try to get from $_POST if not JSON
    //         $webhook_data = (object) $_POST;
    //     }

    //     // Extract order ID and transaction ID
    //     $order_id = null;
    //     $transaction_id = null;
    //     $payment_status = null;

    //     // Check various possible field names based on Placet API
    //     if ( isset( $webhook_data->order_id ) ) {
    //         $order_id = absint( $webhook_data->order_id );
    //     } elseif ( isset( $webhook_data->cart_id ) ) {
    //         $order_id = absint( $webhook_data->cart_id );
    //     }

    //     if ( isset( $webhook_data->transaction_id ) ) {
    //         $transaction_id = sanitize_text_field( $webhook_data->transaction_id );
    //     } elseif ( isset( $webhook_data->token ) ) {
    //         $transaction_id = sanitize_text_field( $webhook_data->token );
    //     }

    //     if ( isset( $webhook_data->status ) ) {
    //         $payment_status = sanitize_text_field( $webhook_data->status );
    //     } elseif ( isset( $webhook_data->payment_status ) ) {
    //         $payment_status = sanitize_text_field( $webhook_data->payment_status );
    //     }

    //     // Validate we have required data
    //     if ( ! $order_id ) {
    //         $this->log( 'Webhook error: No order ID provided' );
    //         status_header( 400 );
    //         exit( 'No order ID' );
    //     }

    //     // Get the order
    //     $order = wc_get_order( $order_id );
        
    //     if ( ! $order ) {
    //         $this->log( 'Webhook error: Order not found: ' . $order_id );
    //         status_header( 404 );
    //         exit( 'Order not found' );
    //     }

    //     // Save transaction ID if provided
    //     if ( $transaction_id ) {
    //         $order->set_transaction_id( $transaction_id );
    //         $order->update_meta_data( '_placet_transaction_id', $transaction_id );
            
    //         $this->log( 'Transaction ID saved from webhook: ' . $transaction_id );
    //     }

    //     // Process based on payment status
    //     if ( $payment_status ) {
    //         switch ( strtolower( $payment_status ) ) {
    //             case 'completed':
    //             case 'success':
    //             case 'paid':
    //             case 'approved':
    //                 // Payment successful
    //                 $order->payment_complete( $transaction_id );
    //                 $order->add_order_note(
    //                     sprintf(
    //                         __( 'Payment completed via Placet Järelmaks. Transaction ID: %s', 'wc-placet-group' ),
    //                         $transaction_id ? $transaction_id : 'N/A'
    //                     )
    //                 );
    //                 $this->log( 'Payment completed for order: ' . $order_id );
    //                 break;

    //             case 'pending':
    //             case 'processing':
    //                 // Payment pending
    //                 $order->update_status( 'on-hold', __( 'Payment pending confirmation from Placet.', 'wc-placet-group' ) );
    //                 $this->log( 'Payment pending for order: ' . $order_id );
    //                 break;

    //             case 'failed':
    //             case 'declined':
    //             case 'cancelled':
    //             case 'rejected':
    //                 // Payment failed
    //                 $order->update_status( 'failed', __( 'Payment failed or was cancelled at Placet.', 'wc-placet-group' ) );
    //                 $this->log( 'Payment failed for order: ' . $order_id );
    //                 break;

    //             default:
    //                 $this->log( 'Unknown payment status: ' . $payment_status );
    //                 $order->add_order_note(
    //                     sprintf(
    //                         __( 'Placet webhook received with status: %s', 'wc-placet-group' ),
    //                         $payment_status
    //                     )
    //                 );
    //         }
    //     } else {
    //         // No status provided, just log the webhook
    //         $order->add_order_note( __( 'Placet webhook received (no status).', 'wc-placet-group' ) );
    //     }

    //     $order->save();

    //     status_header( 200 );
    //     exit( 'OK' );
    // }

    /**
     * Output for the order received page
     *
     * @param int $order_id Order ID.
     */
    public function thankyou_page( $order_id ) {
        $order = wc_get_order( $order_id );

        if ( ! $order ) {
            return;
        }

        // Check for transaction ID in URL parameters (return from payment gateway)
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $transaction_id_from_url = isset( $_GET['transaction_id'] ) ? sanitize_text_field( wp_unslash( $_GET['transaction_id'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( empty( $transaction_id_from_url ) && isset( $_GET['token'] ) ) {
            $transaction_id_from_url = sanitize_text_field( wp_unslash( $_GET['token'] ) );
        }

        // Save transaction ID if provided and not already saved
        if ( ! empty( $transaction_id_from_url ) && ! $order->get_transaction_id() ) {
            $order->set_transaction_id( $transaction_id_from_url );
            $order->update_meta_data( '_placet_transaction_id', $transaction_id_from_url );
            $order->add_order_note(
                sprintf(
                    __( 'Transaction ID received from return URL: %s', 'wc-placet-group' ),
                    $transaction_id_from_url
                )
            );
            $order->save();
            
            // $this->log( 'Transaction ID saved from return URL: ' . $transaction_id_from_url );
        }

        // Check payment status from URL
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $payment_status = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : '';

        // Update order status if payment was successful
        if ( $order->has_status( 'pending' ) || $order->has_status( 'on-hold' ) ) {
            // Check if status indicates success
            if ( in_array( strtolower( $payment_status ), array( 'success', 'completed', 'paid', 'approved' ), true ) ) {
                // Use payment_complete which also triggers emails and reduces stock
                $order->payment_complete( $order->get_transaction_id() );
                
                // $this->log( 'Payment marked as complete for order: ' . $order_id );
            } else {
                // Default behavior - mark as processing
                $order->update_status(
                    'processing',
                    __( 'Payment completed via Placet Järelmaks.', 'wc-placet-group' )
                );
            }
        }

        // Display thank you message
        if ( $this->description ) {
            echo wp_kses_post( wpautop( wptexturize( $this->description ) ) );
        }

        // Display transaction ID if available
        $transaction_id = $order->get_transaction_id();
        if ( $transaction_id ) {
            echo '<p class="woocommerce-notice woocommerce-notice--success woocommerce-thankyou-order-received-transaction-id">';
            echo esc_html(
                sprintf(
                    __( 'Transaction ID: %s', 'wc-placet-group' ),
                    $transaction_id
                )
            );
            echo '</p>';
        }
    }

    /**
     * Handle payment error message display on checkout
     * Shows error notice when customer returns from failed payment
     */
    public function handle_payment_error_message() {
        // Only run on checkout or order-pay pages
        if ( ! is_checkout() && ! is_wc_endpoint_url( 'order-pay' ) ) {
            return;
        }

        // Check if placet_error parameter is present
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( ! isset( $_GET['placet_error'] ) || '1' !== $_GET['placet_error'] ) {
            return;
        }

        // Get the error message from query params (if provided by payment gateway)
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $error_message = isset( $_GET['error_message'] ) ? sanitize_text_field( wp_unslash( $_GET['error_message'] ) ) : '';

        // Use custom error message if provided, otherwise use default
        if ( ! empty( $error_message ) ) {
            $message = sprintf(
                /* translators: %s: Error message from payment gateway */
                __( 'Payment failed: %s. Please try again or choose a different payment method.', 'wc-placet-group' ),
                $error_message
            );
        } else {
            $message = __( 'Payment was cancelled or failed. Please try again or choose a different payment method.', 'wc-placet-group' );
        }

        // Add the error notice
        wc_add_notice( $message, 'error' );

        // Log the error for debugging
        // $this->log( 'Payment error detected. Message: ' . $message );
    }

    /**
     * Logging method
     *
     * @param string $message Log message.
     */
    private function log( $message ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            $logger = wc_get_logger();
            $logger->debug( $message, array( 'source' => 'placet-group' ) );
        }
    }
}

